/**************************************************************************

Copyright (c) 2016, Intel Corporation

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/

#include <Protocol/HiiString.h>
#include <Library/HiiLib.h>

#include "xgbe.h"
#include "DeviceSupport.h"
#include "AdapterInformation.h"
#include "ComponentName.h"
#include "Hii.h"


//
// Global Variables
//
VOID                                    *ixgbe_pxe_memptr = NULL;
PXE_SW_UNDI                             *ixgbe_pxe_31     = NULL; // 3.1 entry
UNDI_PRIVATE_DATA                       *XgbeDeviceList[MAX_NIC_INTERFACES];
NII_TABLE                               ixgbe_UndiData;
UINT8                                   mActiveControllers = 0;
UINT8                                   mActiveChildren    = 0;
EFI_EVENT                               EventNotifyExitBs;
EFI_EVENT                               EventNotifyVirtual;

EFI_HANDLE                              gImageHandle;
EFI_SYSTEM_TABLE                        *gSystemTable;

//
// external Global Variables
//
extern UNDI_CALL_TABLE                     mIxgbeApiTable[];
extern EFI_COMPONENT_NAME_PROTOCOL         gUndiComponentName;
extern EFI_COMPONENT_NAME2_PROTOCOL        gUndiComponentName2;
extern EFI_DRIVER_SUPPORTED_EFI_VERSION_PROTOCOL gUndiSupportedEfiVersion;
extern EFI_DRIVER_DIAGNOSTICS_PROTOCOL     gXgbeUndiDriverDiagnostics;
extern EFI_DRIVER_DIAGNOSTICS2_PROTOCOL    gXgbeUndiDriverDiagnostics2;
extern EFI_DRIVER_HEALTH_PROTOCOL          gUndiDriverHealthProtocol;


extern EFI_GUID                   gEfiStartStopProtocolGuid;

#define EFI_NII_POINTER_PROTOCOL_GUID \
  { \
    0xE3161450, 0xAD0F, 0x11D9, \
    { \
      0x96, 0x69, 0x08, 0x00, 0x20, 0x0c, 0x9a, 0x66 \
    } \
  }

EFI_GUID  gEfiNiiPointerGuid = EFI_NII_POINTER_PROTOCOL_GUID;

//
// function prototypes
//

EFI_STATUS
EFIAPI
InitializeXgbeUNDIDriver (
  IN EFI_HANDLE        ImageHandle,
  IN EFI_SYSTEM_TABLE  *SystemTable
  );

EFI_STATUS
InitInstallConfigTable (
  IN VOID
  );

VOID
EFIAPI
InitUndiNotifyExitBs (
  IN EFI_EVENT Event,
  IN VOID      *Context
  );

EFI_STATUS
InitializePxeStruct (
  VOID
  );

EFI_STATUS
EFIAPI
InitUndiDriverSupported (
  IN EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN EFI_HANDLE                   Controller,
  IN EFI_DEVICE_PATH_PROTOCOL     *RemainingDevicePath
  );

EFI_STATUS
EFIAPI
InitUndiDriverStart (
  IN EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN EFI_HANDLE                   Controller,
  IN EFI_DEVICE_PATH_PROTOCOL     *RemainingDevicePath
  );

EFI_STATUS
EFIAPI
InitUndiDriverStop (
  IN  EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN  EFI_HANDLE                   Controller,
  IN  UINTN                        NumberOfChildren,
  IN  EFI_HANDLE                   *ChildHandleBuffer
  );

EFI_STATUS
InitAppendMac2DevPath (
  IN OUT  EFI_DEVICE_PATH_PROTOCOL   **DevPtr,
  IN      EFI_DEVICE_PATH_PROTOCOL   *BaseDevPtr,
  IN      XGBE_DRIVER_DATA           *XgbeAdapter
  );

VOID
InitUndiPxeStructInit (
  PXE_SW_UNDI *PxePtr,
  UINTN       VersionFlag
  );

VOID
InitUndiPxeUpdate (
  IN XGBE_DRIVER_DATA *NicPtr,
  IN PXE_SW_UNDI      *PxePtr
  );

EFI_STATUS
EFIAPI
UnloadXGigUndiDriver (
  IN EFI_HANDLE ImageHandle
  );

UINT8
CheckSum (
  IN  VOID   *Buffer,
  IN  UINT16 Len
  );

EFI_STATUS
InitPrivateDataAccessProtocol(
    IN UNDI_PRIVATE_DATA     *UndiPrivateData
);

EFI_STATUS
UninstallPrivateDataAccessProtocol(
    IN UNDI_PRIVATE_DATA     *UndiPrivateData
);

EFI_STATUS
InitFcoeConfigurationProtocol(
    IN UNDI_PRIVATE_DATA     *UndiPrivateData
);

EFI_STATUS
UninstallFcoeConfigurationProtocol(
    IN UNDI_PRIVATE_DATA     *UndiPrivateData
);

//
// end function prototypes
//

//
// UNDI Class Driver Global Variables
//
EFI_DRIVER_BINDING_PROTOCOL gUndiDriverBinding = {
  InitUndiDriverSupported,  // Supported
  InitUndiDriverStart,      // Start
  InitUndiDriverStop,       // Stop
  VERSION_TO_HEX,           // Driver Version
  NULL,                     // ImageHandle
  NULL                      // Driver Binding Handle
};


EFI_STATUS
EFIAPI
InitializeXGigUndiDriver(
  IN EFI_HANDLE           ImageHandle,
  IN EFI_SYSTEM_TABLE     *SystemTable
  )
/*++

Routine Description:
  Register Driver Binding protocol for this driver.

Arguments:
  ImageHandle - Handle to this driver image
  SystemTable - Pointer to UEFI system table

Returns:
  EFI_SUCCESS - Driver loaded.
  other       - Driver not loaded.

--*/
{
  EFI_LOADED_IMAGE_PROTOCOL *LoadedImageInterface;
  EFI_STATUS                Status;

  gImageHandle  = ImageHandle;
  gSystemTable  = SystemTable;


  Status = EfiLibInstallDriverBinding (ImageHandle, SystemTable, &gUndiDriverBinding, ImageHandle);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  Status = gBS->InstallMultipleProtocolInterfaces (
                  &gUndiDriverBinding.DriverBindingHandle,
                  &gEfiComponentNameProtocolGuid,
                  &gUndiComponentName,
                  &gEfiComponentName2ProtocolGuid,
                  &gUndiComponentName2,
                  &gEfiDriverDiagnosticsProtocolGuid,
                  &gXgbeUndiDriverDiagnostics,
                  &gEfiDriverDiagnostics2ProtocolGuid,
                  &gXgbeUndiDriverDiagnostics2,
                  &gEfiDriverHealthProtocolGuid,
                  &gUndiDriverHealthProtocol,
                  NULL
                  );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("InstallMultipleProtocolInterfaces returns %x\n", Status));
    return Status;
  }

  if (SystemTable->Hdr.Revision >= EFI_2_10_SYSTEM_TABLE_REVISION) {
    DEBUGPRINT (INIT, ("Installing UEFI 2.1 Supported EFI Version Protocol.\n"));
    Status = gBS->InstallMultipleProtocolInterfaces (
                  &ImageHandle,
                  &gEfiDriverSupportedEfiVersionProtocolGuid,
                  &gUndiSupportedEfiVersion,
                  NULL
                  );
  }
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("InstallMultipleProtocolInterfaces returns %x\n", Status));
    return Status;
  }

  //
  // This protocol does not need to be closed because it uses the GET_PROTOCOL attribute
  //
  Status = gBS->OpenProtocol (
                  ImageHandle,
                  &gEfiLoadedImageProtocolGuid,
                  (VOID *) &LoadedImageInterface,
                  ImageHandle,
                  NULL,
                  EFI_OPEN_PROTOCOL_GET_PROTOCOL
                  );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("%X: OpenProtocol returns %r\n", __LINE__, Status));
    return Status;
  }

  LoadedImageInterface->Unload = UnloadXGigUndiDriver;

  Status = gBS->CreateEvent (
                  EVT_SIGNAL_EXIT_BOOT_SERVICES,
                  TPL_NOTIFY,
                  InitUndiNotifyExitBs,
                  NULL,
                  &EventNotifyExitBs
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("%X: CreateEvent returns %r\n", __LINE__, Status));
    return Status;
  }

  Status = InitializePxeStruct ();

  return Status;
}

EFI_STATUS
InitializePxeStruct (
  VOID
  )
/*++

Routine Description:
  Allocate and initialize both (old and new) the !pxe structures here,
  there should only be one copy of each of these structure for any number
  of NICs this undi supports. Also, these structures need to be on a
  paragraph boundary as per the spec. so, while allocating space for these,
  make sure that there is space for 2 !pxe structures (old and new) and a
  32 bytes padding for alignment adjustment (in case)

Arguments:
  VOID

Returns:
  None

--*/
{
  EFI_STATUS  Status;

  Status = gBS->AllocatePool (
                  EfiBootServicesData,  // EfiRuntimeServicesData,
                  (sizeof (PXE_SW_UNDI) + sizeof (PXE_SW_UNDI) + 32),
                  &ixgbe_pxe_memptr
                  );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("%X: AllocatePool returns %r\n", __LINE__, Status));
    return Status;
  }

  ZeroMem (
    ixgbe_pxe_memptr,
    sizeof (PXE_SW_UNDI) + sizeof (PXE_SW_UNDI) + 32
    );

  //
  // check for paragraph alignment here, assuming that the pointer is
  // already 8 byte aligned.
  //
  if (((UINTN) ixgbe_pxe_memptr & 0x0F) != 0) {
    ixgbe_pxe_31 = (PXE_SW_UNDI *) ((UINTN) ((((UINTN) ixgbe_pxe_memptr) & (0xFFFFFFFFFFFFFFF0)) + 0x10));
  } else {
    ixgbe_pxe_31 = (PXE_SW_UNDI *) ixgbe_pxe_memptr;
  }

  InitUndiPxeStructInit (ixgbe_pxe_31, 0x31); // 3.1 entry
  return Status;
}

VOID
EFIAPI
InitUndiNotifyExitBs (
  IN EFI_EVENT Event,
  IN VOID      *Context
)
/*++

Routine Description:
  When EFI is shuting down the boot services, we need to install a
  configuration table for UNDI to work at runtime!

Arguments:
  (Standard Event handler)

Returns:
  None

--*/
{
  UINT32  i;

  //
  // Divide Active interfaces by two because it tracks both the controller and
  // child handle, then shutdown the receive unit in case it did not get done
  // by the SNP
  //
  for (i = 0; i < mActiveControllers; i++) {
    if (XgbeDeviceList[i]->NicInfo.Hw.device_id != 0) {
      if (XgbeDeviceList[i]->IsChildInitialized) {
        IXGBE_WRITE_REG (&XgbeDeviceList[i]->NicInfo.Hw, IXGBE_RXDCTL (0), 0);
        XgbePciFlush (&XgbeDeviceList[i]->NicInfo);
      }
    }
  }
}

EFI_STATUS
EFIAPI
UnloadXGigUndiDriver (
  IN EFI_HANDLE ImageHandle
  )
/*++

Routine Description:
  Callback to unload the XgbeUndi from memory.

Arguments:
  ImageHandle to driver.

Returns:
  EFI_SUCCESS            - This driver was unloaded successfully.
  EFI_INVALID_PARAMETER  - This driver was not unloaded.

--*/
{
  EFI_HANDLE  *DeviceHandleBuffer;
  UINTN       DeviceHandleCount;
  UINTN       Index;

  EFI_STATUS  Status;

  DEBUGPRINT (INIT, ("XgbeUndiUnload ixgbe_pxe_31->IFcnt = %d\n", ixgbe_pxe_31->IFcnt));
  DEBUGWAIT (INIT);

  //
  // Get the list of all the handles in the handle database.
  // If there is an error getting the list, then the unload operation fails.
  //
  Status = gBS->LocateHandleBuffer (
                  AllHandles,
                  NULL,
                  NULL,
                  &DeviceHandleCount,
                  &DeviceHandleBuffer
                  );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  //
  // Disconnect the driver specified by ImageHandle from all the devices in the
  // handle database.
  //
  DEBUGPRINT (INIT, ("Active interfaces = %d\n", mActiveControllers));
  DEBUGPRINT (INIT, ("Active children = %d\n", mActiveChildren));
  
  for (Index = 0; Index < DeviceHandleCount; Index++) {
    Status = gBS->DisconnectController (
                    DeviceHandleBuffer[Index],
                    ImageHandle,
                    NULL
                    );
  }

  DEBUGPRINT (INIT, ("Active interfaces = %d\n", mActiveControllers));
  DEBUGPRINT (INIT, ("Active children = %d\n", mActiveChildren));

  //
  // Free the buffer containing the list of handles from the handle database
  //
  if (DeviceHandleBuffer != NULL) {
    gBS->FreePool (DeviceHandleBuffer);
  }

  if (mActiveControllers == 0) {
    //
    // Free PXE structures since they will no longer be needed
    //
    Status = gBS->FreePool (ixgbe_pxe_memptr);
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("FreePool returns %x\n", Status));
      return Status;
    }

    //
    // Close both events before unloading
    //
    Status = gBS->CloseEvent (EventNotifyExitBs);
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("CloseEvent EventNotifyExitBs returns %x\n", Status));
      return Status;
    }

    Status = gBS->UninstallMultipleProtocolInterfaces (
                    ImageHandle,
                    &gEfiDriverBindingProtocolGuid,
                    &gUndiDriverBinding,
                    &gEfiComponentNameProtocolGuid,
                    &gUndiComponentName,
                    &gEfiComponentName2ProtocolGuid,
                    &gUndiComponentName2,
                    &gEfiDriverDiagnosticsProtocolGuid,
                    &gXgbeUndiDriverDiagnostics,
                    &gEfiDriverDiagnostics2ProtocolGuid,
                    &gXgbeUndiDriverDiagnostics2,
                    &gEfiDriverHealthProtocolGuid,
                    &gUndiDriverHealthProtocol,
                    NULL
                    );

    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("UninstallMultipleProtocolInterfaces returns %x\n", Status));
      return Status;
    }

    if (gST->Hdr.Revision >= EFI_2_10_SYSTEM_TABLE_REVISION) {
      DEBUGPRINT (INIT, ("Uninstalling UEFI 2.1 Supported EFI Version Protocol.\n"));
      Status = gBS->UninstallMultipleProtocolInterfaces (
                  ImageHandle,
                  &gEfiDriverSupportedEfiVersionProtocolGuid,
                  &gUndiSupportedEfiVersion,
                  NULL
                  );
    }
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("UninstallMultipleProtocolInterfaces returns %x\n", Status));
      return Status;
    }

  } else {
    DEBUGPRINT (INIT, ("Returning EFI_INVALID_PARAMETER\n"));
    DEBUGWAIT (INIT);
    return EFI_INVALID_PARAMETER;
  }

  return Status;
}

/** Checks if device path is not end of device path

   @param[in]  ControllerHandle     Controller handle

   @return     UNDI_PRIVATE_DATA    Pointer to Private Data Structure.
**/
UNDI_PRIVATE_DATA*
GetControllerPrivateData (
  IN  EFI_HANDLE  ControllerHandle
  )
{
  UINT32 i = 0;

  for (i = 0; i < mActiveControllers; i++) {
    if (XgbeDeviceList[i] != NULL) {
      if (XgbeDeviceList[i]->ControllerHandle == ControllerHandle) {
        return XgbeDeviceList[i];
      }
    }
  }
  return NULL;
}

/** Checks if device path is not end of device path

   @param[in]  RemainingDevicePath  Device Path

   @retval     TRUE                 Device path is not end of device path
   @retval     FALSE                Device path is end of device path
**/
BOOLEAN
IsNotEndOfDevicePathNode (
  IN VOID  *RemainingDevicePath
  )
{
  return !(IsDevicePathEnd (RemainingDevicePath));
}

BOOLEAN
IsNullOrEndOfDevicePath (
  IN VOID  *RemainingDevicePath
)
{
  if ((RemainingDevicePath == NULL) || (IsDevicePathEnd (RemainingDevicePath))) {
    return TRUE;
  }
  return FALSE;
}

/** Checks if device path type is supported by the driver

   @param[in]  RemainingDevicePath  Device Path

   @retval     TRUE                 Device path type supported by the driver
   @retval     FALSE                Device path type not supported by the driver
**/
BOOLEAN
IsDevicePathTypeSupported (
  IN VOID  *RemainingDevicePath
  )
{
  UINT8                 PathType;
  UINT8                 PathSubType;
  MAC_ADDR_DEVICE_PATH  *MacDevPath;
  
  if (!RemainingDevicePath) {
    return FALSE;
  }

  PathType    = DevicePathType (RemainingDevicePath);
  PathSubType = DevicePathSubType (RemainingDevicePath);
  
  if ((PathType == MESSAGING_DEVICE_PATH) && (PathSubType == MSG_MAC_ADDR_DP)) {
    MacDevPath = RemainingDevicePath; 
    if (MacDevPath->IfType == PXE_IFTYPE_ETHERNET) { 
      return TRUE;
    }
  }
  return FALSE;
}

/** Checks if device path is supported by the driver

   @param[in]       RemainingDevicePath  Device Path
   @param[in]       MacAddr              MAC Address

   @retval          TRUE                 Device path supported by the driver
   @retval          FALSE                Device path not supported by the driver
**/
#define MAC_ADDRESS_SIZE_IN_BYTES 6
BOOLEAN
IsDevicePathSupported (
  IN VOID   *RemainingDevicePath,
  IN UINT8  *MacAddr
  )
{
  MAC_ADDR_DEVICE_PATH  *MacDevPath;
  UINT8                 Index;

  if(!RemainingDevicePath || !MacAddr) {
    return FALSE;
  }

  if (IsDevicePathTypeSupported (RemainingDevicePath)) {
    MacDevPath = RemainingDevicePath;
    for (Index = 0; Index < MAC_ADDRESS_SIZE_IN_BYTES ; Index++) {
      if (MacDevPath->MacAddress.Addr[Index] != MacAddr[Index]) {
        return FALSE;
      }
    }
    return TRUE;
  }
  return FALSE;
}

/** Analyzes Remaining Device Path

   @param[in]       Controller             Controller Handle
   @param[in]       RemainingDevicePath    Device Path
   @param[in]       VendorId               Vendor Id
   @param[in]       DeviceId               Device Id

   @retval          EFI_SUCCESS            Device supported by the driver
   @retval          EFI_ALREADY_STARTED    Device already managed by the driver
   @retval          EFI_UNSUPPORTED        Device not supported by the driver
**/
EFI_STATUS
AnalyzeRemainingDevicePath (
  IN EFI_HANDLE           Controller,
  IN VOID                 *RemainingDevicePath,
  IN UINT16               VendorId,
  IN UINT16               DeviceId
  )
{
  UNDI_PRIVATE_DATA   *UndiPrivateData;
  
  UndiPrivateData = GetControllerPrivateData(Controller);

  if (UndiPrivateData == NULL) {
    if (IsNullOrEndOfDevicePath(RemainingDevicePath)) {
      if (IsDeviceIdSupported (VendorId, DeviceId)) {
        return EFI_SUCCESS;
      }
    }
    if (IsDevicePathTypeSupported (RemainingDevicePath)) {
      if (IsDeviceIdSupported (VendorId, DeviceId)) {
        return EFI_SUCCESS;
      }
    }
  }

  if (UndiPrivateData != NULL) {
    if (IsNullOrEndOfDevicePath(RemainingDevicePath)) {
      if (UndiPrivateData->IsChildInitialized) {
        return EFI_ALREADY_STARTED;
      }
      else {
        return EFI_SUCCESS;
      }
    }
    if (IsDevicePathSupported (RemainingDevicePath, UndiPrivateData->NicInfo.Hw.mac.addr)) {
      if (UndiPrivateData->IsChildInitialized) {
        return EFI_ALREADY_STARTED;
      } else {
        return EFI_SUCCESS;
      }
    }
  }
  return EFI_UNSUPPORTED;
}

EFI_STATUS
EFIAPI
InitUndiDriverSupported (
  IN EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN EFI_HANDLE                   Controller,
  IN EFI_DEVICE_PATH_PROTOCOL     *RemainingDevicePath
  )
/*++

Routine Description:
  Test to see if this driver supports ControllerHandle. Any ControllerHandle
  than contains a  DevicePath, PciIo protocol, Class code of 2, Vendor ID of 0x8086,
  and DeviceId matching an Intel 10 Gig adapter can be supported.

Arguments:
  This                - Protocol instance pointer.
  Controller          - Handle of device to test.
  RemainingDevicePath - Not used.

Returns:
  EFI_SUCCESS         - This driver supports this device.
  other               - This driver does not support this device.

--*/
{
  EFI_STATUS          Status;
  EFI_PCI_IO_PROTOCOL *PciIo;
  PCI_TYPE00          Pci;

  Status = gBS->OpenProtocol (
                  Controller,
                  &gEfiPciIoProtocolGuid,
                  (VOID **) &PciIo,
                  This->DriverBindingHandle,
                  Controller,
                  EFI_OPEN_PROTOCOL_BY_DRIVER
                  );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (SUPPORTED, ("OpenProtocol1 %r, ", Status));
    return Status;
  }

  Status = PciIo->Pci.Read (
                        PciIo,
                        EfiPciIoWidthUint8,
                        0,
                        sizeof (PCI_CONFIG_HEADER),
                        &Pci
                      );
  if (EFI_ERROR (Status)) {
    goto ExitSupported;
  }

  DEBUGPRINT (SUPPORTED, ("Check devID %X, ", Pci.Hdr.DeviceId));
  
  Status = AnalyzeRemainingDevicePath (
             Controller,
             RemainingDevicePath,
             Pci.Hdr.VendorId,
             Pci.Hdr.DeviceId
           );
  if (EFI_ERROR (Status)) {
    goto ExitSupported;
  }

ExitSupported:
  gBS->CloseProtocol (
        Controller,
        &gEfiPciIoProtocolGuid,
        This->DriverBindingHandle,
        Controller
      );

  return Status;
}

/** Initializes Undi Private Data structure

   @param[in]       Controller             Controller handle
   @param[out]      UndiPrivateData        Driver private data

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_OUT_OF_RESOURCES   Out of memory resources
**/
EFI_STATUS
InitUndiPrivateData (
  IN  EFI_HANDLE                Controller,
  OUT UNDI_PRIVATE_DATA         **UndiPrivateData
  )
{
  UNDI_PRIVATE_DATA *PrivateData;
  
  PrivateData = AllocateZeroPool (sizeof (UNDI_PRIVATE_DATA));
  if (PrivateData == NULL) {
    DEBUGPRINT(CRITICAL, ("AllocateZeroPool returns %r\n", PrivateData));
    DEBUGWAIT(CRITICAL);
    return EFI_OUT_OF_RESOURCES;
  }
  
  PrivateData->Signature               = XGBE_UNDI_DEV_SIGNATURE;
  PrivateData->DeviceHandle            = NULL;
  PrivateData->NicInfo.HwInitialized   = FALSE;
  
  // Save off the controller handle so we can disconnect the driver later
  PrivateData->ControllerHandle = Controller;
  DEBUGPRINT(
    INIT, ("ControllerHandle = %X, DeviceHandle = %X\n",
    PrivateData->ControllerHandle,
    PrivateData->DeviceHandle
  ));
  
  XgbeDeviceList[mActiveControllers] = PrivateData;
  mActiveControllers++;
  
  *UndiPrivateData = PrivateData;
  
  return EFI_SUCCESS;
}

/** Opens controller protocols

   @param[in]       Controller             Controller handle
   @param[in]       This                   Driver Binding Protocol instance
   @param[in]       UndiPrivateData        Driver private data
   @param[out]      UndiDevicePath         Device Path Protocol instance

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
OpenControllerProtocols (
  IN  EFI_HANDLE                   Controller,
  IN  EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN  UNDI_PRIVATE_DATA            *UndiPrivateData
)
{
  EFI_STATUS Status; 
  EFI_DEVICE_PATH_PROTOCOL  *DevicePath;
  
  if ((This == NULL) 
    || (UndiPrivateData == NULL))
  {
    return EFI_INVALID_PARAMETER;
  }
  
  Status = gBS->OpenProtocol (
                  Controller,
                  &gEfiPciIoProtocolGuid,
                  (VOID **) &UndiPrivateData->NicInfo.PciIo,
                  This->DriverBindingHandle,
                  Controller,
                  EFI_OPEN_PROTOCOL_BY_DRIVER
                  );
  if (EFI_ERROR (Status)) {
      DEBUGPRINT(CRITICAL, ("OpenProtocol (EFI PCI IO) returns %r\n", Status));
      DEBUGWAIT(CRITICAL);
      return Status;
  }

  Status = gBS->OpenProtocol (
                  Controller,
                  &gEfiDevicePathProtocolGuid,
                  (VOID **) &DevicePath,
                  This->DriverBindingHandle,
                  Controller,
                  EFI_OPEN_PROTOCOL_BY_DRIVER
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("OpenProtocol (Device Path) returns %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  
  UndiPrivateData->Undi32BaseDevPath = DevicePath;

  return EFI_SUCCESS;
}

/** Initializes controller

   @param[in]       UndiPrivateData        Driver private data

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
   @retval          EFI_OUT_OF_RESOURCES   PCI Init failed
   @retval          EFI_ACCESS_DENIED      Cannot acquire controller
**/
EFI_STATUS
InitController (
  IN UNDI_PRIVATE_DATA  *UndiPrivateData
)
{
  EFI_STATUS Status; 
  
  if (UndiPrivateData == NULL) {
    return EFI_INVALID_PARAMETER;
  }
  // Initialize PCI-E Bus and read PCI related information.
  Status = XgbePciInit (&UndiPrivateData->NicInfo);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("XgbePciInit fails: %r\n", Status));
    return EFI_OUT_OF_RESOURCES;
  }

  // Do all the stuff that is needed when we initialize hw for the first time
  Status = XgbeFirstTimeInit (UndiPrivateData);
  if (EFI_ERROR (Status)
    && (Status != EFI_ACCESS_DENIED))
  {
    DEBUGPRINT (CRITICAL, ("XgbeFirstTimeInit fails: %r\n", Status));
    return Status;
  }

  if (Status == EFI_ACCESS_DENIED) {
    UndiPrivateData->NicInfo.UNDIEnabled = FALSE;
  } else {
    UndiPrivateData->NicInfo.UNDIEnabled = TRUE;
  }
  
  UndiPrivateData->AltMacAddrSupported = IsAltMacAddrSupported(UndiPrivateData);
  
  return EFI_SUCCESS;
}

/** Executes Configuration Protocols

   @param[in]   UndiPrivateData        Driver private data
   @param[in]   This                   Driver binding protocol instance
   @param[in]   Controller             Controller handle

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
ExecuteConfigurationProtocols (
  IN UNDI_PRIVATE_DATA            *UndiPrivateData,
  IN EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN EFI_HANDLE                   Controller
)
{
  return EFI_SUCCESS;
}

/** Initializes Network Interface Identifier Pointer Protocol

   @param[in]       Handle              Controller/Child handle
   @param[in]       NiiProtocol31      NII Protocol instance
   @param[out]      NIIPointerProtocol  NII Pointer Protocol instance
   
   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitNiiPointerProtocol (
  IN   EFI_HANDLE                                 *Handle,
  IN   EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL  *NiiProtocol31,
  OUT  EFI_NII_POINTER_PROTOCOL                   *NIIPointerProtocol
  )
{
  EFI_STATUS  Status;
  
  if ((Handle == NULL) 
    || (NiiProtocol31 == NULL)
    || (NIIPointerProtocol == NULL))
  {
    return EFI_INVALID_PARAMETER;
  }
  
  NIIPointerProtocol->NiiProtocol31 = NiiProtocol31;

  Status = gBS->InstallMultipleProtocolInterfaces (
                  Handle,
                  &gEfiNiiPointerGuid,
                  NIIPointerProtocol,
                  NULL
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InstallMultipleProtocolInterfaces gEfiNiiPointerGuid returns %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  
  return EFI_SUCCESS;
}

/** Initializes controller protocols

   @param[in]       UndiPrivateData        Driver private data
   @param[in]       Controller             Controller handle

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitControllerProtocols (
  IN UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN EFI_HANDLE         Controller
)
{
  EFI_STATUS Status; 
  
  if (UndiPrivateData == NULL) {
    return EFI_INVALID_PARAMETER;
  }
  
  ComponentNameInitializeControllerName (UndiPrivateData);

  // The EFI_NII_POINTER_PROTOCOL protocol is used only by this driver.  It is done so that
  // we can get the NII protocol from either the parent or the child handle.  This is convenient
  // in the Diagnostic protocol because it allows the test to be run when called from either the
  // parent or child handle which makes it more user friendly.
  Status = InitNiiPointerProtocol (
             &Controller, 
             &UndiPrivateData->NiiProtocol31,
             &UndiPrivateData->NIIPointerProtocol
           );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InitNiiPointerProtocol returned %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  
  return EFI_SUCCESS;
}

/** Initializes Undi Callback functions in Adapter structure. 

    @param[out]      NicInfo    Adapter Structure which shall be initialized

    @return          NicInfo    Initialized adapter structure
**/
VOID
InitUndiCallbackFunctions (
  OUT XGBE_DRIVER_DATA  *NicInfo
  )
{
  // Initialize the UNDI callback functions to 0 so that the default boot services
  // callback is used instead of the SNP callback.
  NicInfo->Delay       = (VOID *) 0;
  NicInfo->Virt2Phys   = (VOID *) 0;
  NicInfo->Block       = (VOID *) 0;
  NicInfo->Map_Mem     = (VOID *) 0;
  NicInfo->UnMap_Mem   = (VOID *) 0;
  NicInfo->Sync_Mem    = (VOID *) 0;
  NicInfo->Unique_ID   = (UINT64) NicInfo;
  NicInfo->VersionFlag = 0x31;
}

/** Initializes UNDI (PXE) structures

   @param[in]       UndiPrivateData        Private data structure

   @retval          None
**/
VOID
InitUndiStructures (
  IN UNDI_PRIVATE_DATA  *UndiPrivateData
)
{
  // the IfNum index for the current interface will be the total number
  // of interfaces initialized so far
  InitUndiPxeUpdate(&UndiPrivateData->NicInfo, ixgbe_pxe_31);
  InitUndiCallbackFunctions (&UndiPrivateData->NicInfo);
}

/** Initializes Device Path Protocol

   @param[in]       UndiPrivateData        Driver private data

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitDevicePathProtocol (
  IN   UNDI_PRIVATE_DATA         *UndiPrivateData
)
{
  EFI_STATUS Status;
  
  if (UndiPrivateData == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  Status = InitAppendMac2DevPath (
             &UndiPrivateData->Undi32DevPath,
             UndiPrivateData->Undi32BaseDevPath,
             &UndiPrivateData->NicInfo
           );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("Could not append mac address to the device path.  Error = %r\n", Status));
    DEBUGWAIT (CRITICAL);
    return Status;
  }
  
  Status = gBS->InstallMultipleProtocolInterfaces (
                  &UndiPrivateData->DeviceHandle,
                  &gEfiDevicePathProtocolGuid, 
                  UndiPrivateData->Undi32DevPath,
                  NULL
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InstallMultipleProtocolInterfaces gEfiDevicePathProtocolGuid returns %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  return EFI_SUCCESS;
}

/** Initializes Driver Stop Protocol

   @param[in]       UndiPrivateData        Driver private data 

   
   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitDriverStopProtocol (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData
)
{
  EFI_STATUS Status;
  
  if (UndiPrivateData == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  UndiPrivateData->DriverStop.StartDriver = StartDriver;
  UndiPrivateData->DriverStop.StopDriver = StopDriver;
  
  Status = gBS->InstallMultipleProtocolInterfaces (
                  &UndiPrivateData->DeviceHandle,
                  &gEfiStartStopProtocolGuid,
                  &UndiPrivateData->DriverStop,
                  NULL
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InstallMultipleProtocolInterfaces returns %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  return EFI_SUCCESS;
}
/** Initializes Network Interface Identifier Protocol

   @param[in]       Handle           Controller/Child handle
   @param[out]      NiiProtocol31   NII Protocol instance
   
   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitNiiProtocol (
  IN   EFI_HANDLE                                 *Handle,
  OUT  EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL  *NiiProtocol31
  )
{
  EFI_STATUS  Status;


  NiiProtocol31->Id            = (UINT64) (ixgbe_pxe_31);
  
  // IFcnt should be equal to the total number of physical ports - 1
  NiiProtocol31->IfNum         = ixgbe_pxe_31->IFcnt;
  NiiProtocol31->Revision      = EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_REVISION;
  NiiProtocol31->Type          = EfiNetworkInterfaceUndi;
  NiiProtocol31->MajorVer      = PXE_ROMID_MAJORVER;
  NiiProtocol31->MinorVer      = PXE_ROMID_MINORVER_31;
  NiiProtocol31->ImageSize     = 0;
  NiiProtocol31->ImageAddr     = 0;
  NiiProtocol31->Ipv6Supported = TRUE;

  NiiProtocol31->StringId[0]   = 'U';
  NiiProtocol31->StringId[1]   = 'N';
  NiiProtocol31->StringId[2]   = 'D';
  NiiProtocol31->StringId[3]   = 'I';
  
  
  Status = gBS->InstallMultipleProtocolInterfaces (
                  Handle,
                  &gEfiNetworkInterfaceIdentifierProtocolGuid_31, 
                  NiiProtocol31,
                  NULL
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InstallMultipleProtocolInterfaces gEfiNetworkInterfaceIdentifierProtocolGuid_31 returns %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }
  return EFI_SUCCESS;
}
/** Initializes child protocols

   @param[in]       UndiPrivateData        Driver private data

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
InitChildProtocols (
  IN UNDI_PRIVATE_DATA         *UndiPrivateData
)
{
  EFI_STATUS Status;
  
  if (UndiPrivateData == NULL) {
    return EFI_INVALID_PARAMETER;
  }
  
  Status = InitDriverStopProtocol (UndiPrivateData);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InitDriverStopProtocol returned %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }

  Status = InitDevicePathProtocol (
             UndiPrivateData
           );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InitDevicePathProtocol returned %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }

  Status = InitNiiPointerProtocol (
             &UndiPrivateData->DeviceHandle, 
             &UndiPrivateData->NiiProtocol31,
             &UndiPrivateData->NIIPointerProtocol
             );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("InitNiiPointerProtocol returned %r\n", Status));
    DEBUGWAIT(CRITICAL);
    return Status;
  }

  if (UndiPrivateData->NicInfo.UNDIEnabled) {
    Status = InitNiiProtocol (
               &UndiPrivateData->DeviceHandle,
               &UndiPrivateData->NiiProtocol31
               );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT(CRITICAL, ("InitNiiProtocol returned %r\n", Status));
      DEBUGWAIT(CRITICAL);
      return Status;
    }
  }


  Status = InitAdapterInformationProtocol (UndiPrivateData);
  if (EFI_ERROR (Status) 
    && (Status != EFI_UNSUPPORTED))
  {
    DEBUGPRINT (CRITICAL, ("Could not install Adapter Information protocol interface - %r\n", Status));
    return Status;
  }


  // HII may not install, so we do not want to return any errors
  Status = HiiInit (UndiPrivateData);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("HiiInit returns %r\n", Status));
  }


  return EFI_SUCCESS;
}

/** Opens protocols for Child device

   @param[in]       UndiPrivateData        Driver private data
   @param[in]       This                   Driver Binding protocol instance
   @param[in]       Controller             Controller handle

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
   @retval          EFI_DEVICE_ERROR       Failed to open PCI IO protocol
**/
EFI_STATUS
OpenChildProtocols (
  IN  UNDI_PRIVATE_DATA            *UndiPrivateData,
  IN  EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN  EFI_HANDLE                   Controller
)
{
  EFI_STATUS                  Status;
  EFI_PCI_IO_PROTOCOL         *PciIo;
  
  if ((UndiPrivateData == NULL )
    || (This == NULL)) 
  {
    return EFI_INVALID_PARAMETER;
  }

  // Open For Child Device
  Status = gBS->OpenProtocol (
              Controller,
              &gEfiPciIoProtocolGuid,
              (VOID **) &PciIo,
              This->DriverBindingHandle,
              UndiPrivateData->DeviceHandle,
              EFI_OPEN_PROTOCOL_BY_CHILD_CONTROLLER
              );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("OpenProtocol PciIo returned %r\n", Status));
    return Status;
  }

  return EFI_SUCCESS;
}

/** Closes controller protocols

   @param[in]       Controller             Controller handle
   @param[in]       This                   Driver Binding protocol instance

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
CloseControllerProtocols (
  IN EFI_HANDLE                   Controller,
  IN EFI_DRIVER_BINDING_PROTOCOL  *This
)
{
  EFI_STATUS Status;
  
  if ((Controller == NULL)
    || (This == NULL))
  {
    return EFI_INVALID_PARAMETER;
  }
  
  Status = gBS->CloseProtocol (
             Controller,
             &gEfiDevicePathProtocolGuid,
             This->DriverBindingHandle,
             Controller
             );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("CloseProtocol gEfiDevicePathProtocolGuid returned %r\n", Status));
    return Status;
  }

  Status = gBS->CloseProtocol (
             Controller,
             &gEfiPciIoProtocolGuid,
             This->DriverBindingHandle,
             Controller
             );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("CloseProtocol gEfiPciIoProtocolGuid returned %r\n", Status));
    return Status;
  }
  return EFI_SUCCESS;
}

EFI_STATUS
EFIAPI
InitUndiDriverStart (
  IN EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN EFI_HANDLE                   Controller,
  IN EFI_DEVICE_PATH_PROTOCOL     *RemainingDevicePath
  )
/*++

Routine Description:
  Start this driver on Controller by opening PciIo and DevicePath protocol.
  Initialize PXE structures, create a copy of the Controller Device Path with the
  NIC's MAC address appended to it, install the NetworkInterfaceIdentifier protocol
  on the newly created Device Path.

Arguments:
  This                - Protocol instance pointer.
  Controller          - Handle of device to work with.
  RemainingDevicePath - Not used, always produce all possible children.

Returns:
  EFI_SUCCESS         - This driver is added to Controller.
  other               - This driver does not support this device.

--*/
{
  UNDI_PRIVATE_DATA         *XgbePrivate         = NULL;
  EFI_STATUS                Status;
  BOOLEAN                   InitializeChild      = TRUE;
  BOOLEAN                   InitializeController = TRUE;

  DEBUGPRINT (INIT, ("XgbeUndiDriverStart\n"));
  DEBUGWAIT (INIT);

  XgbePrivate = GetControllerPrivateData (Controller);
  if (XgbePrivate != NULL) {
    InitializeController = FALSE;
  }

  if (RemainingDevicePath != NULL) {
    InitializeChild = IsNotEndOfDevicePathNode(RemainingDevicePath);
  }

  if (!InitializeController 
    && !InitializeChild)
  {
    return EFI_SUCCESS;
  }

  if (InitializeController) {
    
    Status = InitUndiPrivateData (
               Controller,
               &XgbePrivate
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT(CRITICAL, ("InitUndiPrivateData returns %r\n", Status));
      DEBUGWAIT(CRITICAL);
      goto UndiError;
    }
  
    Status = OpenControllerProtocols (
               Controller,
               This,
               XgbePrivate
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT(CRITICAL, ("OpenControllerProtocols returns %r\n", Status));
      DEBUGWAIT(CRITICAL);
      goto UndiError;
    }
    
    Status = InitController(XgbePrivate);
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("InitController fails: %r\n", Status));
      goto UndiError;
    }
  
    Status = ExecuteConfigurationProtocols (
               XgbePrivate,
               This,
               Controller
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("ExecuteConfigurationProtocols failed with %r\n", Status));
      goto UndiError;
    }
  
    Status = InitControllerProtocols (
               XgbePrivate,
               Controller
             );  
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("InitControllerProtocols failed with %r\n", Status));
      goto UndiError;
    }
  }
    // CHILD INIT
  if (InitializeChild) {
    InitUndiStructures(XgbePrivate);
             
    Status = InitChildProtocols(
               XgbePrivate
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("InitChildProtocols failed with %r\n", Status));
      goto UndiErrorDeleteDevicePath;
    }
  
    Status = OpenChildProtocols (
               XgbePrivate,
               This,
               Controller
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("OpenChildProtocols failed with %r\n", Status));
    }
  
    XgbePrivate->IsChildInitialized = TRUE;
  }
  DEBUGPRINT (INIT, ("XgbeUndiDriverStart - success\n"));
  
  return EFI_SUCCESS;

UndiErrorDeleteDevicePath:
  gBS->FreePool (XgbePrivate->Undi32DevPath);
  InitUndiPxeUpdate (NULL, ixgbe_pxe_31);

UndiError:
  XgbeDeviceList[mActiveControllers - 1] = NULL;
  mActiveControllers--;
  
  CloseControllerProtocols (
    Controller,
    This
  );

  gBS->FreePool ((VOID *) XgbePrivate);

  DEBUGPRINT (INIT, ("XgbeUndiDriverStart - error %x\n", Status));
  
  return Status;
}

/** Stops controller

   @param[in]       This                   Driver Binding protocol instance
   @param[in]       Controller             Controller handle

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
StopController (
  IN  EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN  EFI_HANDLE                   Controller,
  IN  UNDI_PRIVATE_DATA            *UndiPrivateData
  )
{
  EFI_STATUS                Status;

  if (This == NULL
    || UndiPrivateData == NULL)
  {
    return EFI_INVALID_PARAMETER;
  }
  DEBUGPRINT(CRITICAL, ("Entering Stop Controller"));

  DEBUGPRINT(INIT, ("Uninstalling NIIPointerProtocol protocol\n"));
  Status = gBS->UninstallMultipleProtocolInterfaces (
                  Controller,
                  &gEfiNiiPointerGuid,
                  &UndiPrivateData->NIIPointerProtocol,
                  NULL
                );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("UninstallMultipleProtocolInterfaces gEfiNiiPointerGuid failed - %r\n", Status));
    // This one should be always installed so there is real issue if we cannot uninstall
    return Status;
  }

  DEBUGPRINT (INIT, ("EfiLibFreeUnicodeStringTable"));
  FreeUnicodeStringTable (UndiPrivateData->ControllerNameTable);

  Status = UndiPrivateData->NicInfo.PciIo->FreeBuffer (
                                             UndiPrivateData->NicInfo.PciIo,
                                             UNDI_MEM_PAGES (MEMORY_NEEDED),
                                             (VOID *) (UINTN) UndiPrivateData->NicInfo.MemoryPtr
                                             );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("PCI IO FreeBuffer returns %X\n", Status));
    DEBUGWAIT (INIT);
    return Status;
  }

  DEBUGPRINT (INIT, ("Attributes"));
  Status = UndiPrivateData->NicInfo.PciIo->Attributes (
                                             UndiPrivateData->NicInfo.PciIo,
                                             EfiPciIoAttributeOperationDisable,
                                             EFI_PCI_DEVICE_ENABLE | EFI_PCI_IO_ATTRIBUTE_DUAL_ADDRESS_CYCLE,
                                             NULL
                                             );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("2. PCI IO Attributes returns %X\n", Status));
    DEBUGWAIT (INIT);
    return Status;
  }
  // The below is commmented out because it causes a crash when SNP, MNP, and ARP drivers are loaded
  // This has not been root caused but it is probably because some driver expects IFcnt not to change
  // This should be okay because when ifCnt is set when the driver is started it is based on ActiveInterfaces
  //  InitUndiPxeUpdate (NULL, ixgbe_pxe);
  //  InitUndiPxeUpdate (NULL, ixgbe_pxe_31);
  Status = CloseControllerProtocols (
             Controller,
             This
           );

  XgbeDeviceList[UndiPrivateData->NiiProtocol31.IfNum] = NULL;
  
  DEBUGPRINT (INIT, ("FreePool(UndiPrivateData->Undi32DevPath)"));
  Status = gBS->FreePool (UndiPrivateData->Undi32DevPath);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("FreePool(UndiPrivateData->Undi32DevPath) returns %r\n", Status));
  }
  
  Status = gBS->FreePool (UndiPrivateData);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(INIT, ("FreePool(UndiPrivateData) returns %r\n", Status));
    DEBUGWAIT (INIT);
  }
  
  return EFI_SUCCESS;
}

/** Stops child

   @param[in]       This                   Driver Binding protocol instance
   @param[in]       Controller             Controller handle
   @param[in]       ChildHandleBuffer      Buffer with child handles

   @retval          EFI_SUCCESS            Procedure returned successfully
   @retval          EFI_INVALID_PARAMETER  Invalid parameter passed
**/
EFI_STATUS
StopChild (
  IN  EFI_DRIVER_BINDING_PROTOCOL  *This,
  IN  EFI_HANDLE                   Controller,
  IN  EFI_HANDLE                   *ChildHandleBuffer,
  IN  UNDI_PRIVATE_DATA            *UndiPrivateData
  )
{
  EFI_STATUS                Status;

  if (This == NULL 
    || ChildHandleBuffer == NULL
    || UndiPrivateData == NULL)
  {
    return EFI_INVALID_PARAMETER;
  }

  Status = UninstallAdapterInformationProtocol(UndiPrivateData);

  // Close the bus driver
  DEBUGPRINT (INIT, ("removing gEfiPciIoProtocolGuid\n"));
  Status = gBS->CloseProtocol (
                  Controller,
                  &gEfiPciIoProtocolGuid,
                  This->DriverBindingHandle,
                  ChildHandleBuffer[0]
                  );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("Close of gEfiPciIoProtocolGuid failed with %r\n", Status));
    DEBUGWAIT (INIT);
    return Status;
  }

  DEBUGPRINT (INIT, ("%d: UninstallMultipleProtocolInterfaces\n", __LINE__));
  if (UndiPrivateData->NicInfo.UNDIEnabled) {
      Status = gBS->UninstallMultipleProtocolInterfaces (
                      UndiPrivateData->DeviceHandle,
                      &gEfiStartStopProtocolGuid,
                      &UndiPrivateData->DriverStop,
                      &gEfiNetworkInterfaceIdentifierProtocolGuid_31,
                      &UndiPrivateData->NiiProtocol31,
                      &gEfiNiiPointerGuid,
                      &UndiPrivateData->NIIPointerProtocol,
                      &gEfiDevicePathProtocolGuid,
                      UndiPrivateData->Undi32DevPath,
                      NULL
                      );
  } else {
      Status = gBS->UninstallMultipleProtocolInterfaces (
                      UndiPrivateData->DeviceHandle,
                      &gEfiStartStopProtocolGuid,
                      &UndiPrivateData->DriverStop,
                      &gEfiNiiPointerGuid,
                      &UndiPrivateData->NIIPointerProtocol,
                      &gEfiDevicePathProtocolGuid,
                      UndiPrivateData->Undi32DevPath,
                      NULL
                      );
  }
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("1. UninstallMultipleProtocolInterfaces returns %r\n", Status));
    DEBUGWAIT (INIT);
  }

  // If we get the ACCESS_DENIED status code usually calling UninstallMultipleProtocolInterfaces a second
  // time will uninstall the protocols successfully.
  if (Status == EFI_ACCESS_DENIED) {
    DEBUGPRINT (INIT, ("%d: UninstallMultipleProtocolInterfaces\n", __LINE__));
    if (UndiPrivateData->NicInfo.UNDIEnabled) {
        Status = gBS->UninstallMultipleProtocolInterfaces (
                        UndiPrivateData->DeviceHandle,
                        &gEfiStartStopProtocolGuid,
                        &UndiPrivateData->DriverStop,
                        &gEfiNetworkInterfaceIdentifierProtocolGuid_31,
                        &UndiPrivateData->NiiProtocol31,
                        &gEfiNiiPointerGuid,
                        &UndiPrivateData->NIIPointerProtocol,
                        &gEfiDevicePathProtocolGuid,
                        UndiPrivateData->Undi32DevPath,
                        NULL
                        );
    } else {
        Status = gBS->UninstallMultipleProtocolInterfaces (
                        UndiPrivateData->DeviceHandle,
                        &gEfiStartStopProtocolGuid,
                        &UndiPrivateData->DriverStop,
                        &gEfiNiiPointerGuid,
                        &UndiPrivateData->NIIPointerProtocol,
                        &gEfiDevicePathProtocolGuid,
                        UndiPrivateData->Undi32DevPath,
                        NULL
                        );
    }
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (INIT, ("1. UninstallMultipleProtocolInterfaces returns %r\n", Status));
      DEBUGWAIT (INIT);
    }
  }
  
  Status = HiiUnload (UndiPrivateData);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("HiiUnload returns %r\n", Status));
  }
  

  if (UndiPrivateData->NicInfo.UNDIEnabled) {
    XgbeShutdown (&UndiPrivateData->NicInfo);
    XgbeClearRegBits (&UndiPrivateData->NicInfo, IXGBE_CTRL_EXT, IXGBE_CTRL_EXT_DRV_LOAD);
  }

  return EFI_SUCCESS;
}

EFI_STATUS
EFIAPI
InitUndiDriverStop (
  IN EFI_DRIVER_BINDING_PROTOCOL    *This,
  IN EFI_HANDLE                     Controller,
  IN UINTN                          NumberOfChildren,
  IN EFI_HANDLE                     *ChildHandleBuffer
  )
/*++

Routine Description:
  Stop this driver on Controller by removing NetworkInterfaceIdentifier protocol and
  closing the DevicePath and PciIo protocols on Controller.

Arguments:
  This              - Protocol instance pointer.
  Controller        - Handle of device to stop driver on.
  NumberOfChildren  - How many children need to be stopped.
  ChildHandleBuffer - Child handle buffer to uninstall.

Returns:
  EFI_SUCCESS       - This driver is removed Controller.
  EFI_DEVICE_ERROR  - The driver could not be successfully stopped.
  other             - This driver was not removed from this device.

--*/
{
  EFI_STATUS                Status;
  UNDI_PRIVATE_DATA         *XgbePrivate;
  EFI_NII_POINTER_PROTOCOL  *NIIPointerProtocol;

  DEBUGPRINT (INIT, ("XgbeUndiDriverStop\n"));
  DEBUGWAIT (INIT);

  // Open an instance for the Network Interface Identifier Protocol so we can check to see
  // if the interface has been shutdown.  Does not need to be closed because we use the
  // GET_PROTOCOL attribute to open it.
  Status = gBS->OpenProtocol (
                  Controller,
                  &gEfiNiiPointerGuid,
                  (VOID **) &NIIPointerProtocol,
                  This->DriverBindingHandle,
                  Controller,
                  EFI_OPEN_PROTOCOL_GET_PROTOCOL
                  );

  if (EFI_ERROR (Status)) {
    DEBUGPRINT (INIT, ("%d: OpenProtocol returns %r\n", __LINE__, Status));
    return Status;
  }

  XgbePrivate = UNDI_PRIVATE_DATA_FROM_THIS (NIIPointerProtocol->NiiProtocol31);
  DEBUGPRINT (INIT, ("State = %X\n", XgbePrivate->NicInfo.State));
  DEBUGWAIT (INIT);

  // If we are called with less than one child handle it means that we already sucessfully
  // uninstalled
  DEBUGPRINT (INIT, ("Number of children %d\n", NumberOfChildren));
  if (NumberOfChildren == 0) {
    Status = StopController (
               This,
               Controller, 
               XgbePrivate
               );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT(CRITICAL, ("StopController failed with status: %r\n", Status));
      return EFI_DEVICE_ERROR;
    }

    // Decrement the number of controllers this driver is supporting
    mActiveControllers--;

    return EFI_SUCCESS;
  }

  if (NumberOfChildren > 1) {
    DEBUGPRINT(INIT, ("Unexpected number of child handles.\n"));
    return EFI_OUT_OF_RESOURCES;
  }
  
  Status = StopChild (
             This,
             Controller,
             ChildHandleBuffer,
             XgbePrivate
             );
  if (EFI_ERROR (Status)) {
    DEBUGPRINT(CRITICAL, ("StopChild failed with status: %r\n", Status));
    return EFI_DEVICE_ERROR;
  }
  
  mActiveChildren--;
  
  return Status;
}

VOID
WaitForEnter (
  VOID
  )
/*++

Routine Description:
  This is only for debugging, it will pause and wait for the user to press <ENTER>
  Results AFTER this call are unpredicable. You can only be assured the code up to
  this call is working.

Arguments:
  VOID

Returns:
  none

--*/
{
  EFI_INPUT_KEY Key;

  AsciiPrint ("\nPress <Enter> to continue...\n");

  do {
    gST->ConIn->ReadKeyStroke (gST->ConIn, &Key);
  } while (Key.UnicodeChar != 0xD);
}

EFI_STATUS
InitAppendMac2DevPath (
  IN OUT EFI_DEVICE_PATH_PROTOCOL  **DevPtr,
  IN EFI_DEVICE_PATH_PROTOCOL      *BaseDevPtr,
  IN XGBE_DRIVER_DATA              *XgbeAdapter
  )
/*++

Routine Description:
  Using the NIC data structure information, read the EEPROM to get the MAC address and then allocate space
  for a new devicepath (**DevPtr) which will contain the original device path the NIC was found on (*BaseDevPtr)
  and an added MAC node.

Arguments:
  DevPtr            - Pointer which will point to the newly created device path with the MAC node attached.
  BaseDevPtr        - Pointer to the device path which the UNDI device driver is latching on to.
  XgbeAdapter   - Pointer to the NIC data structure information which the UNDI driver is layering on..

Returns:
  EFI_SUCCESS       - A MAC address was successfully appended to the Base Device Path.
  other             - Not enough resources available to create new Device Path node.

--*/
{
  MAC_ADDR_DEVICE_PATH      MacAddrNode;
  EFI_DEVICE_PATH_PROTOCOL  *EndNode;
  UINT16                    i;
  UINT16                    TotalPathLen;
  UINT16                    BasePathLen;
  EFI_STATUS                Status;
  UINT8                     *DevicePtr;

  DEBUGPRINT (INIT, ("XgbeAppendMac2DevPath\n"));

  ZeroMem ((char *) &MacAddrNode, sizeof (MacAddrNode));
  CopyMem (
    (char *) &MacAddrNode.MacAddress,
    (char *) XgbeAdapter->Hw.mac.perm_addr,
    IXGBE_ETH_LENGTH_OF_ADDRESS
    );

  DEBUGPRINT (INIT, ("\n"));
  for (i = 0; i < 6; i++) {
    DEBUGPRINT (INIT, ("%2x ", MacAddrNode.MacAddress.Addr[i]));
  }

  DEBUGPRINT (INIT, ("\n"));
  for (i = 0; i < 6; i++) {
    DEBUGPRINT (INIT, ("%2x ", XgbeAdapter->Hw.mac.perm_addr[i]));
  }

  DEBUGPRINT (INIT, ("\n"));
  DEBUGWAIT (INIT);

  MacAddrNode.Header.Type       = MESSAGING_DEVICE_PATH;
  MacAddrNode.Header.SubType    = MSG_MAC_ADDR_DP;
  MacAddrNode.Header.Length[0]  = sizeof (MacAddrNode);
  MacAddrNode.Header.Length[1]  = 0;
  MacAddrNode.IfType            = PXE_IFTYPE_ETHERNET;

  //
  // find the size of the base dev path.
  //
  EndNode = BaseDevPtr;
  while (!IsDevicePathEnd (EndNode)) {
    EndNode = NextDevicePathNode (EndNode);
  }

  BasePathLen = (UINT16) ((UINTN) (EndNode) - (UINTN) (BaseDevPtr));

  //
  // create space for full dev path
  //
  TotalPathLen = (UINT16) (BasePathLen + sizeof (MacAddrNode) + sizeof (EFI_DEVICE_PATH_PROTOCOL));

  Status = gBS->AllocatePool (
                  EfiBootServicesData,  // EfiRuntimeServicesData,
                  TotalPathLen,
                  &DevicePtr
                  );

  if (Status != EFI_SUCCESS) {
    return Status;
  }

  //
  // copy the base path, mac addr and end_dev_path nodes
  //
  *DevPtr = (EFI_DEVICE_PATH_PROTOCOL *) DevicePtr;
  CopyMem (DevicePtr, (char *) BaseDevPtr, BasePathLen);
  DevicePtr += BasePathLen;
  CopyMem (DevicePtr, (char *) &MacAddrNode, sizeof (MacAddrNode));
  DevicePtr += sizeof (MacAddrNode);
  CopyMem (DevicePtr, (char *) EndNode, sizeof (EFI_DEVICE_PATH_PROTOCOL));

  return EFI_SUCCESS;
}

VOID
InitUndiPxeUpdate (
  IN XGBE_DRIVER_DATA *NicPtr,
  IN PXE_SW_UNDI      *PxePtr
  )
/*++

Routine Description:
  When called with a null NicPtr, this routine decrements the number of NICs
  this UNDI is supporting and removes the NIC_DATA_POINTER from the array.
  Otherwise, it increments the number of NICs this UNDI is supported and
  updates the pxe.Fudge to ensure a proper check sum results.

Arguments:
    NicPtr = contains bus, dev, function etc.
    PxePtr = Pointer to the PXE structure

Returns:
  None

--*/
{
  if (NicPtr == NULL) {
    //
    // IFcnt is equal to the number of NICs this undi supports - 1
    //
    if (mActiveChildren > 0) {
      mActiveChildren--;
    }
  } else {
    mActiveChildren++;
  }

  //
  // number of NICs this undi supports
  //
  PxePtr->IFcnt = mActiveChildren - 1;
  PxePtr->Fudge = (UINT8) (PxePtr->Fudge - CheckSum ((VOID *) PxePtr, PxePtr->Len));
  DEBUGPRINT (INIT, ("XgbeUndiPxeUpdate: ActiveChildren = %d\n", mActiveChildren));
  DEBUGPRINT (INIT, ("XgbeUndiPxeUpdate: PxePtr->IFcnt = %d\n", PxePtr->IFcnt));
}

VOID
InitUndiPxeStructInit (
  PXE_SW_UNDI *PxePtr,
  UINTN       VersionFlag
  )
/*++

Routine Description:
  Initialize the !PXE structure

Arguments:
    PxePtr = Pointer to the PXE structure to initialize
    VersionFlag = Indicates PXE version 3.0 or 3.1

Returns:
  EFI_SUCCESS         - This driver is added to Controller.
  other               - This driver does not support this device.

--*/
{
  //
  // initialize the !PXE structure
  //
  PxePtr->Signature = PXE_ROMID_SIGNATURE;
  PxePtr->Len       = sizeof (PXE_SW_UNDI);
  PxePtr->Fudge     = 0;  // cksum
  PxePtr->IFcnt     = 0;  // number of NICs this undi supports
  PxePtr->Rev       = PXE_ROMID_REV;
  PxePtr->MajorVer  = PXE_ROMID_MAJORVER;
  PxePtr->MinorVer  = PXE_ROMID_MINORVER_31;
  PxePtr->reserved1 = 0;

  PxePtr->Implementation = PXE_ROMID_IMP_SW_VIRT_ADDR |
    PXE_ROMID_IMP_FRAG_SUPPORTED |
    PXE_ROMID_IMP_CMD_LINK_SUPPORTED |
    PXE_ROMID_IMP_NVDATA_READ_ONLY |
    PXE_ROMID_IMP_STATION_ADDR_SETTABLE |
    PXE_ROMID_IMP_PROMISCUOUS_MULTICAST_RX_SUPPORTED |
    PXE_ROMID_IMP_PROMISCUOUS_RX_SUPPORTED |
    PXE_ROMID_IMP_BROADCAST_RX_SUPPORTED |
    PXE_ROMID_IMP_FILTERED_MULTICAST_RX_SUPPORTED |
    PXE_ROMID_IMP_SOFTWARE_INT_SUPPORTED |
    PXE_ROMID_IMP_PACKET_RX_INT_SUPPORTED;

  PxePtr->EntryPoint    = (UINT64) UndiApiEntry;
  PxePtr->MinorVer      = PXE_ROMID_MINORVER_31;

  PxePtr->reserved2[0]  = 0;
  PxePtr->reserved2[1]  = 0;
  PxePtr->reserved2[2]  = 0;
  PxePtr->BusCnt        = 1;
  PxePtr->BusType[0]    = PXE_BUSTYPE_PCI;

  PxePtr->Fudge         = (UINT8) (PxePtr->Fudge - CheckSum ((VOID *) PxePtr, PxePtr->Len));

  //
  // return the !PXE structure
  //
}

UINT8
CheckSum (
  IN VOID   *Buffer,
  IN UINT16 Len
  )
/*++

Routine Description:
  This does an 8 bit check sum of the passed in buffer for Len bytes.
  This is primarily used to update the check sum in the SW UNDI header.

Arguments:
  Buffer         - Pointer to the passed in buffer to check sum
  Len            - Length of buffer to be check summed in bytes.

Returns:
  The 8-bit checksum of the array pointed to by buf.

--*/
{
  UINT8 Chksum;
  INT8  *Bp;

  Chksum = 0;

  if ((Bp = Buffer) != NULL) {
    while (Len--) {
      Chksum = (UINT8) (Chksum +*Bp++);

    }

  }

  return Chksum;
}


EFI_STATUS
UndiGetControllerHealthStatus (
  IN  UNDI_PRIVATE_DATA              *UndiPrivateData,
  OUT EFI_DRIVER_HEALTH_STATUS       *DriverHealthStatus,
  OUT EFI_DRIVER_HEALTH_HII_MESSAGE  **MessageList
)
/*++

Routine Description:
  Return the health status of the controller. If there is a message status that
  is related to the current health status it is also prepared and returned
  by this function

Arguments:
  UndiPrivateData      - controller data
  DriverHealthStatus   - controller status to be returned
  MessageList          - pointer to the message list to be returned

Returns:
  EFI_STATUS

--*/
{
  EFI_STATUS                     Status;
  EFI_DRIVER_HEALTH_HII_MESSAGE  *ErrorMessage;


  BOOLEAN                        SfpModuleQualified;
  UINT32                         ErrorCount;

  DEBUGPRINT(HEALTH, ("\n"));

  if ((UndiPrivateData == NULL) || (DriverHealthStatus == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  if (MessageList != NULL) {
    *MessageList = NULL;
  }

  *DriverHealthStatus = EfiDriverHealthStatusHealthy;
  SfpModuleQualified  = TRUE;
  ErrorCount = 0;
  //
  // Check if module is supported/qualified for media types fiber and unknown.
  //
  if (UndiPrivateData->NicInfo.Hw.phy.media_type == ixgbe_media_type_fiber ||
      UndiPrivateData->NicInfo.Hw.phy.media_type == ixgbe_media_type_unknown) {
    if (!IsQualifiedSfpModule(&UndiPrivateData->NicInfo)) {
      DEBUGPRINT(HEALTH, ("*DriverHealthStatus = EfiDriverHealthStatusFailed\n"));
      *DriverHealthStatus = EfiDriverHealthStatusFailed;
      SfpModuleQualified  = FALSE;
      ErrorCount++;
    }
  }

  if (*DriverHealthStatus == EfiDriverHealthStatusHealthy) {
    return EFI_SUCCESS;
  }

  //
  // Create error message string
  //
  if ((MessageList == NULL) || (UndiPrivateData->HiiHandle == NULL)) {
    //
    // Text message are not requested or HII is not supported on this port
    //
    return EFI_SUCCESS;
  }

  //
  // Need to allocate space for two message entries:
  // the one for the message we need to pass to UEFI BIOS and
  // one for NULL entry indicating the end of list
  //
  Status = gBS->AllocatePool (
                  EfiBootServicesData,
                  (ErrorCount + 1) * sizeof (EFI_DRIVER_HEALTH_HII_MESSAGE),
                  (VOID **) MessageList
                  );
  if (EFI_ERROR (Status)) {
    *MessageList = NULL;
    return EFI_OUT_OF_RESOURCES;
  }

  ErrorCount = 0;
  ErrorMessage = *MessageList;

  if (!SfpModuleQualified) {
    Status = HiiSetString (
               UndiPrivateData->HiiHandle,
               STRING_TOKEN(STR_DRIVER_HEALTH_MESSAGE),
               L"The driver failed to load because an unsupported module type was detected.",
               NULL // All languages will be updated
               );
    if (EFI_ERROR (Status)) {
      *MessageList = NULL;
      return EFI_OUT_OF_RESOURCES;
    }
    ErrorMessage[ErrorCount].HiiHandle = UndiPrivateData->HiiHandle;
    ErrorMessage[ErrorCount].StringId = STRING_TOKEN(STR_DRIVER_HEALTH_MESSAGE);
    ErrorCount++;
  }

  //
  // Indicate the end of list by setting HiiHandle to NULL
  //
  ErrorMessage[ErrorCount].HiiHandle = NULL;
  ErrorMessage[ErrorCount].StringId = 0;

  return EFI_SUCCESS;
}

EFI_STATUS
UndiGetDriverHealthStatus (
  OUT EFI_DRIVER_HEALTH_STATUS       *DriverHealthStatus
)
/*++

Routine Description:
  Return the cumulative health status of all controllers managed by the driver

Arguments:
  DriverHealthStatus - controller status to be returned

Returns:
  EFI_STATUS

--*/
{
  EFI_STATUS                 Status;
  EFI_DRIVER_HEALTH_STATUS   HealthStatus;
  UINTN                      i;

  DEBUGPRINT(HEALTH, ("\n"));

  if (DriverHealthStatus == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  HealthStatus = EfiDriverHealthStatusHealthy;
  *DriverHealthStatus = EfiDriverHealthStatusHealthy;

  //
  // Iterate throug all controllers managed by this instance of driver and
  // ask them about their health status
  //
  for (i = 0; i < mActiveControllers; i++) {
    if (XgbeDeviceList[i]->NicInfo.Hw.device_id != 0) {
      Status = UndiGetControllerHealthStatus(XgbeDeviceList[i], &HealthStatus, NULL);
      if (EFI_ERROR (Status)) {
        DEBUGPRINT(CRITICAL, ("UndiGetHealthStatus - %r\n"));
        return EFI_DEVICE_ERROR;
      }
      if (HealthStatus != EfiDriverHealthStatusHealthy) {
        *DriverHealthStatus = EfiDriverHealthStatusFailed;
      }
    }
  }

  return EFI_SUCCESS;
}
